<?php
/**
 * @package     Joomla.Site
 * @subpackage  mod_agriculteur_liste
 *
 * @copyright   Copyright (C) 2025 Votre Nom. All rights reserved.
 * @license     GNU General Public License version 2 or later; see LICENSE.txt
 */

defined('_JEXEC') or die;

use Joomla\CMS\Uri\Uri;
use Joomla\CMS\Router\Route;
use Joomla\CMS\Factory;

// Récupérer l'ID unique du module
$moduleId = $module->id;

// Récupérer l'application
$app = Factory::getApplication();

// Paramètres de pagination
$itemsPerPage = (int)$params->get('items_per_page', 5); // Nombre d'agriculteurs par page
$currentPage = max(1, $app->input->getInt('page', 1)); // Page actuelle
$startIndex = ($currentPage - 1) * $itemsPerPage;
$totalItems = count($agriculteurs);
$totalPages = ceil($totalItems / $itemsPerPage);

// Détecter la vue active (liste ou grille) à partir des paramètres d'URL
$activeView = $app->input->getCmd('view_mode', 'list'); // Par défaut: liste

// Récupérer la catégorie à filtrer depuis l'URL
$filterCategory = $app->input->getString('category', '');

// Sous-ensemble d'agriculteurs pour la page actuelle
$pageItems = array_slice($agriculteurs, $startIndex, $itemsPerPage);

// Construire l'URL pour la pagination avec SP Page Builder et préserver la vue active et la catégorie
$queryParams = [
    'option' => 'com_sppagebuilder',
    'view' => 'page',
    'id' => $params->get('sppb_page_id', '50'),
    'view_mode' => $activeView
];

// Ajouter le paramètre de catégorie s'il existe
if (!empty($filterCategory)) {
    $queryParams['category'] = $filterCategory;
}

$root = Uri::root();
$baseUrl = $root . 'index.php?' . http_build_query(array_merge(
    $queryParams,
    array_diff_key($_GET, array_flip(['option', 'view', 'id', 'view_mode', 'category', 'page']))
));

// URL spécifique à votre site pour les fiches agriculteurs
$profileUrlBase = $root . 'index.php/nos-agriculteurs';

// Inclure le fichier CSS externe
$document = Factory::getDocument();
$document->addStyleSheet($root . 'modules/mod_agriculteur_liste/tmpl/assets/styles.css');

// Ajouter des variables JavaScript pour le module
$document->addScriptDeclaration("
    var moduleId = '$moduleId';
    var activeView = '$activeView';
    var currentPage = $currentPage;
    var profileUrlBase = '$profileUrlBase';
    var root = '" . $root . "';
");

// Ajouter le script de débogage directement dans le template
$debugScriptPath = $root . 'debug-geoloc.js';
$document->addScript($debugScriptPath . '?v=' . time()); // Ajouter un paramètre pour éviter la mise en cache
?>

<!-- Script de débogage inline pour s'assurer qu'il s'exécute -->
<script>
console.log("=== SCRIPT DE DÉBOGAGE INLINE CHARGÉ ===");
console.log("Module ID:", "<?php echo $moduleId; ?>");
console.log("Mode d'affichage:", "<?php echo $activeView; ?>");
console.log("Page actuelle:", <?php echo $currentPage; ?>);
</script>

<div class="agriculteur-liste-container" id="agriculteur-liste-<?php echo $moduleId; ?>">
    <!-- Barre de navigation supérieure -->
    <div class="top-navigation-bar">
        <!-- Boutons de navigation -->
        <div class="nav-buttons">
            <button id="map-view-button" class="nav-button <?php echo $activeView === 'map' ? 'active' : ''; ?>">
                <i class="fas fa-map-marked-alt"></i>
            </button>
            <button id="grid-view-button" class="nav-button <?php echo $activeView === 'grid' ? 'active' : ''; ?>">
                <i class="fas fa-th"></i>
            </button>
            <button id="search-button" class="nav-button">
                <i class="fas fa-search"></i>
            </button>
        </div>
        
        <!-- Bouton de géolocalisation -->
        <button id="geolocate-button-<?php echo $moduleId; ?>" class="geolocation-button">
            <i class="fas fa-location-crosshairs"></i> Les plus proches
        </button>
    </div>
    
    <!-- Information sur les distances -->
    <div id="distances-info-<?php echo $moduleId; ?>" class="distances-info" style="display: none;">
        <i class="fas fa-info-circle"></i> Les distances routières sont calculées à partir de votre position.
    </div>
    
    <!-- Conteneur de la carte -->
    <div id="map-container" class="map-container" style="display: <?php echo $activeView === 'map' ? 'block' : 'none'; ?>;">
        <div id="google-map" class="google-map"></div>
    </div>
    
    <!-- Conteneur de la grille/liste -->
    <div id="grid-container" class="grid-container" style="display: <?php echo $activeView !== 'map' ? 'block' : 'none'; ?>;">
        <!-- Vue liste -->
        <div id="list-view" class="list-view <?php echo $activeView === 'list' ? 'active-view' : ''; ?>">
            <ul id="farmer-list-<?php echo $moduleId; ?>" class="farmer-list">
                <?php foreach ($pageItems as $agriculteur) : ?>
                    <?php
                    // Construire l'URL du profil
                    $profileUrl = $profileUrlBase . '?submission_id=' . $agriculteur['id'] . '&page_retour=' . $currentPage . '&view_mode=' . $activeView;
                    
                    // Formater le nom complet
                    $ownerName = trim($agriculteur['prenom'] . ' ' . $agriculteur['nom']);
                    
                    // Extraire les coordonnées
                    $coordinates = $agriculteur['coordinates'];
                    ?>
                    <li>
                        <a href="<?php echo $profileUrl; ?>" class="farmer-item" 
                           data-id="<?php echo $agriculteur['id']; ?>"
                           data-name="<?php echo htmlspecialchars($agriculteur['societe']); ?>"
                           data-owner="<?php echo htmlspecialchars($ownerName); ?>"
                           data-coordinates="<?php echo htmlspecialchars($coordinates); ?>">
                            <div class="farmer-content">
                                <img src="<?php echo $agriculteur['image']; ?>" alt="<?php echo htmlspecialchars($agriculteur['societe']); ?>" class="farmer-image">
                                <div class="farmer-info">
                                    <h3 class="farmer-name"><?php echo htmlspecialchars($agriculteur['societe']); ?></h3>
                                    <p class="farmer-owner"><?php echo htmlspecialchars($ownerName); ?></p>
                                </div>
                                <div class="farmer-distance" id="distance-<?php echo $agriculteur['id']; ?>">
                                    <p class="distance-value">0.0</p>
                                    <p class="distance-unit">km</p>
                                </div>
                            </div>
                        </a>
                    </li>
                <?php endforeach; ?>
            </ul>
        </div>
        
        <!-- Vue grille -->
        <div id="grid-view" class="grid-view <?php echo $activeView === 'grid' ? 'active-view' : ''; ?>">
            <div id="farmer-grid-<?php echo $moduleId; ?>" class="farmer-grid">
                <?php foreach ($pageItems as $agriculteur) : ?>
                    <?php
                    // Construire l'URL du profil
                    $profileUrl = $profileUrlBase . '?submission_id=' . $agriculteur['id'] . '&page_retour=' . $currentPage . '&view_mode=' . $activeView;
                    
                    // Formater le nom complet
                    $ownerName = trim($agriculteur['prenom'] . ' ' . $agriculteur['nom']);
                    
                    // Extraire les coordonnées
                    $coordinates = $agriculteur['coordinates'];
                    ?>
                    <a href="<?php echo $profileUrl; ?>" class="grid-item" 
                       data-id="<?php echo $agriculteur['id']; ?>"
                       data-name="<?php echo htmlspecialchars($agriculteur['societe']); ?>"
                       data-owner="<?php echo htmlspecialchars($ownerName); ?>"
                       data-coordinates="<?php echo htmlspecialchars($coordinates); ?>">
                        <div class="grid-image-container">
                            <img src="<?php echo $agriculteur['image']; ?>" alt="<?php echo htmlspecialchars($agriculteur['societe']); ?>" class="grid-image">
                        </div>
                        <div class="grid-content">
                            <h3 class="grid-name"><?php echo htmlspecialchars($agriculteur['societe']); ?></h3>
                            <p class="grid-owner"><?php echo htmlspecialchars($ownerName); ?></p>
                            <div class="grid-distance" id="grid-distance-<?php echo $agriculteur['id']; ?>">
                                <p class="grid-distance-value">0.0</p>
                                <p class="grid-distance-unit">km</p>
                            </div>
                        </div>
                    </a>
                <?php endforeach; ?>
            </div>
        </div>
        
        <!-- Conteneur des résultats de recherche -->
        <div id="search-results-container" class="search-results-container" style="display: none;">
            <h2 id="search-count" class="search-count">Résultats de recherche</h2>
            <ul id="search-results-list" class="search-results-list"></ul>
        </div>
    </div>
    
    <!-- Pagination -->
    <?php if ($totalPages > 1) : ?>
    <div class="pagination-container">
        <div class="page-info">
            Page <?php echo $currentPage; ?> sur <?php echo $totalPages; ?>
        </div>
        <div class="pagination">
            <?php if ($currentPage > 1) : ?>
                <a href="<?php echo $baseUrl . '&page=1'; ?>" class="page-item first">
                    <i class="fas fa-angle-double-left"></i>
                </a>
                <a href="<?php echo $baseUrl . '&page=' . ($currentPage - 1); ?>" class="page-item prev">
                    <i class="fas fa-chevron-left"></i>
                </a>
            <?php endif; ?>
            
            <?php
            // Calculer les pages à afficher
            $maxPagesToShow = 5;
            $startPage = max(1, $currentPage - floor($maxPagesToShow / 2));
            $endPage = min($totalPages, $startPage + $maxPagesToShow - 1);
            
            // Ajuster startPage si nécessaire
            $startPage = max(1, $endPage - $maxPagesToShow + 1);
            
            // Afficher la première page si nécessaire
            if ($startPage > 1) {
                echo '<a href="' . $baseUrl . '&page=1' . '" class="page-item">1</a>';
                
                // Ajouter des points de suspension si nécessaire
                if ($startPage > 2) {
                    echo '<span class="page-item ellipsis">...</span>';
                }
            }
            
            // Afficher les numéros de page
            for ($i = $startPage; $i <= $endPage; $i++) {
                $class = ($i == $currentPage) ? 'page-item active' : 'page-item';
                echo '<a href="' . $baseUrl . '&page=' . $i . '" class="' . $class . '">' . $i . '</a>';
            }
            
            // Afficher la dernière page si nécessaire
            if ($endPage < $totalPages) {
                // Ajouter des points de suspension si nécessaire
                if ($endPage < $totalPages - 1) {
                    echo '<span class="page-item ellipsis">...</span>';
                }
                
                echo '<a href="' . $baseUrl . '&page=' . $totalPages . '" class="page-item">' . $totalPages . '</a>';
            }
            ?>
            
            <?php if ($currentPage < $totalPages) : ?>
                <a href="<?php echo $baseUrl . '&page=' . ($currentPage + 1); ?>" class="page-item next">
                    <i class="fas fa-chevron-right"></i>
                </a>
            <?php endif; ?>
        </div>
    </div>
    <?php endif; ?>
    
    <!-- Popup de recherche -->
    <div id="search-popup" class="search-popup">
        <div class="search-popup-content">
            <div class="search-header">
                <h2>Rechercher un agriculteur</h2>
                <button id="search-close" class="search-close">×</button>
            </div>
            <div class="search-form">
                <input type="text" id="search-input" class="search-input" placeholder="Nom, produits, catégories...">
                <button id="search-submit" class="search-submit">Rechercher</button>
            </div>
        </div>
    </div>
</div>
